/*
 * Project:         XYZ
 * Script Purpose:  This adapter is developed to push updates from OBM Event to relevant Opsgenie Alert. It updates alert with SNOW Ticket details
 * Developer:       Machani Vidyasagar
 * Last Update:     Mar 2024

 * Script LogFile:  /opt/HP/BSM/log/opr/integration/OpsgenieBacksyncAdapter.log
 */

import com.hp.opr.api.Version
import com.hp.opr.api.ws.adapter.ForwardChangeArgs
import com.hp.opr.api.ws.adapter.ForwardEventArgs
import com.hp.opr.api.ws.adapter.GetExternalEventArgs
import com.hp.opr.api.ws.adapter.InitArgs
import com.hp.opr.api.ws.adapter.PingArgs
import com.hp.opr.api.ws.adapter.ReceiveChangeArgs
import com.hp.opr.api.ws.adapter.BulkReceiveChangeArgs

import com.hp.opr.api.ws.model.event.OprAnnotation
import com.hp.opr.api.ws.model.event.OprAnnotationList
import com.hp.opr.api.ws.model.event.OprControlTransferInfo
import com.hp.opr.api.ws.model.event.OprControlTransferStateEnum
import com.hp.opr.api.ws.model.event.OprCustomAttribute
import com.hp.opr.api.ws.model.event.OprCustomAttributeList
import com.hp.opr.api.ws.model.event.OprEvent
import com.hp.opr.api.ws.model.event.OprEventList
import com.hp.opr.api.ws.model.event.OprEventChange
import com.hp.opr.api.ws.model.event.OprEventReference
import com.hp.opr.api.ws.model.event.OprGroup
import com.hp.opr.api.ws.model.event.OprPriority
import com.hp.opr.api.ws.model.event.OprSeverity
import com.hp.opr.api.ws.model.event.OprState
import com.hp.opr.api.ws.model.event.OprSymptomList
import com.hp.opr.api.ws.model.event.OprSymptomReference
import com.hp.opr.api.ws.model.event.OprUser

import com.hp.opr.api.ws.model.event.ci.OprConfigurationItem
import com.hp.opr.api.ws.model.event.ci.OprForwardingInfo
import com.hp.opr.api.ws.model.event.ci.OprForwardingTypeEnum
import com.hp.opr.api.ws.model.event.ci.OprNodeReference
import com.hp.opr.api.ws.model.event.ci.OprRelatedCi

import com.hp.opr.api.ws.model.event.property.OprAnnotationPropertyChange
import com.hp.opr.api.ws.model.event.property.OprCustomAttributePropertyChange
import com.hp.opr.api.ws.model.event.property.OprEventPropertyChange
import com.hp.opr.api.ws.model.event.property.OprEventPropertyNameEnum
import com.hp.opr.api.ws.model.event.property.OprGroupPropertyChange
import com.hp.opr.api.ws.model.event.property.OprIntegerPropertyChange
import com.hp.opr.api.ws.model.event.property.OprPropertyChangeOperationEnum
import com.hp.opr.api.ws.model.event.property.OprSymptomPropertyChange
import com.hp.opr.api.ws.model.event.property.OprUserPropertyChange

import com.hp.opr.common.model.NodeInfo
import com.hp.opr.common.util.BsmEnvironment
import com.hp.opr.eventsync.ws.client.WinkClientSupport
import com.hp.opr.platform.api.config.SettingsManagerConfiguration

import com.hp.ucmdb.api.UcmdbService
import com.hp.ucmdb.api.UcmdbServiceFactory
import com.hp.ucmdb.api.UcmdbServiceProvider
import com.hp.ucmdb.api.topology.QueryDefinition
import com.hp.ucmdb.api.topology.QueryLink
import com.hp.ucmdb.api.topology.QueryNode
import com.hp.ucmdb.api.topology.Topology
import com.hp.ucmdb.api.topology.TopologyCount
import com.hp.ucmdb.api.topology.TopologyQueryService
import com.hp.ucmdb.api.topology.TopologyUpdateFactory
import com.hp.ucmdb.api.topology.indirectlink.IndirectLink
import com.hp.ucmdb.api.topology.indirectlink.IndirectLinkStepToPart
import com.hp.ucmdb.api.types.TopologyCI
import com.hp.ucmdb.api.types.UcmdbId

import groovy.util.slurpersupport.GPathResult
import groovy.xml.MarkupBuilder
import java.security.cert.X509Certificate
import java.text.SimpleDateFormat
import javax.ws.rs.core.Cookie
import javax.ws.rs.core.MediaType
import javax.ws.rs.core.MultivaluedMap
import javax.xml.bind.JAXBElement

import org.apache.commons.codec.binary.Base64
import org.apache.wink.client.ClientRequest
import org.apache.wink.client.ClientResponse
import org.apache.wink.client.ClientWebException
import org.apache.wink.client.Resource
import org.apache.wink.client.RestClient

import java.io.File
import java.util.List
import java.util.Date

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

class OpsgenieBacksyncAdapter
{
  static def ROOT_DRILLDOWN_PATH = "/opr-console/opr-evt-details?eventId="
  static def LOG_DIR_REL = "log${File.separator}opr${File.separator}integration"
  static def LOGFILE_NAME = "OpsgenieBacksyncAdapter.log"

  def m_logfileDir = null
  def m_logfile = null
  def m_logger = null
  def m_initArgs = null

  def init(def args)
  {
    m_logger = args.logger
    m_initArgs = args

    m_logfileDir = new File("${args.installDir}${File.separator}${LOG_DIR_REL}")
    m_logfile = new File(m_logfileDir, LOGFILE_NAME)

    m_logger.info("Logfile Adapter initalized. INSTALL_DIR=${args.installDir}")

    appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
    def timestamp = new Date()
    def msg = """\n### ${timestamp.toString()}: init() called ###
    parameter connected server ID: ${m_initArgs.connectedServerId}
    parameter connected server name: ${m_initArgs.connectedServerName}
    parameter connected server display name: ${m_initArgs.connectedServerDisplayName}\n"""
    appendLogfile(msg)
    appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
  }

  def destroy()
  {
    m_logger.info("Logfile Adapter destroy.")

    def timestamp = new Date()
    def msg = """\n### ${timestamp.toString()}: destroy() called ###\n"""
    appendLogfile(msg)
    appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
  }

  def ping(def args)
  {
    args.outputDetail = "Success."
    return true
  }

  def forwardEvent(def args)
  {
    def timestamp = new Date()
    /* appendLogfile("\n### ${timestamp.toString()}: forwardEvent() called ###") */
    args.externalRefId = logEvent(args.event, args.info, args.causeExternalRefId, args.credentials?.userName)
    // Make a drilldown base path to the original event as an example
    args.drilldownUrlPath = "${ROOT_DRILLDOWN_PATH}${args.event.getId()}"

    logRelatedCI(args.event.relatedCi)

    boolean includeReferences = true
    OprEvent event = args.getEvent(args.event.id, includeReferences)
    String msg = """\n\tevent history lines count: ${event?.historyListRef?.historyLineList?.historyLines?.size()}"""
    /* appendLogfile(msg) */

    return true
  }

  private def logEvent(def event, def info, def causeExternalRefId, def userName)
  {
    def extId = "urn:uuid:${event.getId()}"

    def controlServerName = (event.controlTransferredTo) ? event.controlTransferredTo.name : "<none>"
    def controlServerDnsName = (event.controlTransferredTo) ? event.controlTransferredTo.dnsName : ""
    def controlTransferredState = (event.controlTransferredTo) ? event.controlTransferredTo.state : "<none>"
    def controlExternalId = (event.controlTransferredTo) ? event.controlTransferredTo.externalId : ""

    def msg = """
    event.id: ${event.id}
    event.title: ${event.title}
    event.state: ${event.state}
    forwarding.type: ${info.forwardingType}
    event.time.received: ${event.timeReceived}
    event.time.first.received: ${event.timeFirstReceived}
    event.control.transferred.to.name: ${controlServerName}
    event.control.transferred.to.dns.name: ${controlServerDnsName}
    event.control.transferred.to.state: ${controlTransferredState}
    event.control.transferred.to.external.id: ${controlExternalId}
    event.duplicate_count: ${event.duplicateCount}
    event.external.id: ${extId}
    cause.external.id: ${causeExternalRefId}
    target login username: ${userName}"""
    /* appendLogfile(msg) */

    return extId
  }

  def logRelatedCI(def relatedCi)
  {
    if (relatedCi == null || relatedCi.configurationItem == null)
      return

    OprConfigurationItem ci = relatedCi.configurationItem
    Map<String, Object> properties = ci.properties
    if (properties == null || properties.isEmpty())
      return
  }

  def forwardEvents(def args)
  {
    def timestamp = new Date()
    /* appendLogfile("\n### ${timestamp.toString()}: forwardEvents() called ###") */
    OprEventList events = args.events

    /* appendLogfile("\n*** Begin Bulk Forward ***\n") */
    for (OprEvent event in events.eventList)
    {
      def causeExternalRefId = getCauseExternalRefId(args, event)
      def externalRefId = logEvent(event, event.getForwardingInfo(m_initArgs.connectedServerId),
                                   causeExternalRefId, args.credentials?.userName)
      // Make a drilldown base path to the original event as an example
      def drilldownUrlPath = "${ROOT_DRILLDOWN_PATH}${event.id}"

      logRelatedCI(event.relatedCi)
      args.setForwardSuccess(event.id, externalRefId, drilldownUrlPath)
    }
    /* appendLogfile("\n*** End Bulk Forward ***\n") */
    return true
  }

  def getCauseExternalRefId(def args, def event)
  {
    String causeExternalRefId = ""

    if (event?.cause?.targetId)
    {
      OprEvent causeEvent = args.getEvent(event.cause.targetId, false)
      if (causeEvent)
      {
        OprForwardingInfo causeInfo = causeEvent.getForwardingInfo(m_initArgs.connectedServerId)
        causeExternalRefId = causeInfo?.externalId
      }
    }
    return causeExternalRefId
  }
  
  
  def forwardChange(def args)
  {
    def timestamp = new Date()
    def ticketFlag = true
    def apiKey = args.credentials?.password
    appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter forwardChange() called ###\n")

    args.changes.changedProperties.each 
    {
      def propertyChange ->
      if ((propertyChange.propertyName.equals("custom_attribute") && propertyChange.key.toLowerCase().contains("tsticket")) || (propertyChange.propertyName.equals("state") && propertyChange.currentValue.equals("closed")))
			{
				final OprEvent event = args.getEvent()
        //final OprEvent event = args.getEvent(propertyChange.eventId, true)
        //final OprEvent event = args.getEvent(args.change.eventId, true)
        //final OprEvent event = args.getEvent(args.change.eventId ?: args.change.id, true)

        if (ticketFlag)
        {
          if (propertyChange.propertyName.equals("custom_attribute") && propertyChange.key.toLowerCase().contains("tsticket"))
          {
            appendLogfile("\n[INFO]: ${timestamp.toString()}: EventId: ${event.getId().toString()}: ${propertyChange.propertyName} changed.\n")
            appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
            updateOpsgenieIncident(args.event, apiKey)
            ticketFlag = false
          }
          else
          {
            appendLogfile("\n[INFO]: ${timestamp.toString()}: EventId: ${event.getId().toString()}: ${propertyChange.propertyName} changed.\n")
            appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
            updateOpsgenieState(args.event, apiKey)
            ticketFlag = false
          }
        }
			}
    }
    return true
  }

  def forwardChanges(def args)
	{
		def timestamp = new Date()
    appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter forwardChanges() called ###\n")
    def apiKey = args.credentials?.password
  
		for (OprEventChange change in args.changes.eventChanges)
		{
      def ticketFlag = true
			change.changedProperties.each 
			{
				def propertyChange ->
        if ((propertyChange.propertyName.equals("custom_attribute") && propertyChange.key.toLowerCase().contains("tsticket")) || (propertyChange.propertyName.equals("state") && propertyChange.currentValue.equals("closed")))
			  {
          final OprEvent event = args.getEvent()
          //final OprEvent event = args.getEvent(propertyChange.eventId, true)
          if (ticketFlag)
          {
            if (propertyChange.propertyName.equals("custom_attribute") && propertyChange.key.toLowerCase().contains("tsticket"))
            {
              ticketFlag = false
              args.setForwardSuccess(change.id)
            }
            else
            {
              ticketFlag = false
              args.setForwardSuccess(change.id)
            }
          }
			  }
			}
		}
    return true
	}

  def updateOpsgenieIncident(def event, def apiKey)
  {
    /* https://community.microfocus.com/it_ops_mgt/ops_bdg/f/itrc-162/348722/event-getcustomattribute-and-event-getnode-do-not-work-in-the-logfileadapter-sample-script */
    def timestamp = new Date()
    String eventId = "${event.id.toString()}"
    String opsgenieAlertId = "${event.getType().toString()}"
    String tsTicket = ""
    String tsTicketAssignment = ""
    String tsTicketLink = ""
    String tsTicketPriority = ""
    String tsTicketUrgency = ""
    def tsTicketFlag = true
    def tsTicketAssignmentFlag = true
    def tsTicketLinkFlag = true
    def tsTicketPriorityFlag = true
    def tsTicketUrgencyFlag = true

    appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter updateOpsgenieIncident() called ###\n")
    appendLogfile("\nOpsgenie alertId: ${opsgenieAlertId}\n")

    if (event.customAttributes?.customAttributes?.size()) 
    {
      event.customAttributes.customAttributes.each 
      {
        ca ->
        if (ca.name.trim() == "TsTicket")
        {
          if(tsTicketFlag)
          {
            tsTicket = "${ca.value}";
            tsTicketFlag = false; 
          }
        }
        if (ca.name.trim() == "TsTicketAssignment")
        {
          if(tsTicketAssignmentFlag)
          {
            tsTicketAssignment = "${ca.value}";
            tsTicketAssignmentFlag = false; 
          }
        }
        if (ca.name.trim() == "TsTicketLink")
        {
          if(tsTicketLinkFlag)
          {
            tsTicketLink = "${ca.value}";
            tsTicketLinkFlag = false; 
          }
        }
        if (ca.name.trim() == "TsTicketPriority")
        {
          if(tsTicketPriorityFlag)
          {
            tsTicketPriority = "${ca.value}";
            tsTicketPriorityFlag = false; 
          }
        }
        if (ca.name.trim() == "TsTicketUrgency")
        {
          if(tsTicketUrgencyFlag)
          {
            tsTicketUrgency = "${ca.value}";
            tsTicketUrgencyFlag = false; 
          }
        }
      }
    }
    appendLogfile("\nIncident details of event: ${eventId}")
    appendLogfile("\n cma TsTicket: ${tsTicket}")
    appendLogfile("\n cma TsTicketAssignment: ${tsTicketAssignment}")
    appendLogfile("\n cma TsTicketLink: ${tsTicketLink}")
    appendLogfile("\n cma TsTicketPriority: ${tsTicketPriority}")
    appendLogfile("\n cma TsTicketUrgency: ${tsTicketUrgency}")
    appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")

    try 
    {
      String genericURL = "https://api.eu.opsgenie.com/v2/alerts/alert_id/notes?identifierType=id";
      
      genericURL = genericURL.replace('alert_id', opsgenieAlertId)

      URL url = new URL(genericURL);
      Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress("1.2.3.4", 8080));
      HttpURLConnection conn = (HttpURLConnection) url.openConnection(proxy);

      conn.setRequestProperty("Authorization", "GenieKey " + "${apiKey.toString()}");
      conn.setRequestMethod("POST");
      conn.setRequestProperty("Content-Type", "application/json; utf-8");
      conn.setDoOutput(true);

      String urlParameters = """{
        "user": "Smile Webhook",
        "source": "OBM",
        "note": "OBM Event Id: eventId \\n  Ticket Number: tsTicketValue \\n  Ticket Assignment Group: tsTicketAssignmentValue \\n  Ticket URL: tsTicketLinkValue \\n  Ticket Priority: tsTicketPriorityValue \\n  Ticket Urgency: tsTicketUrgencyValue"
      }"""

      urlParameters = urlParameters.replace('eventId', eventId)
      urlParameters = urlParameters.replace('tsTicketValue', tsTicket)
      urlParameters = urlParameters.replace('tsTicketAssignmentValue', tsTicketAssignment)
      urlParameters = urlParameters.replace('tsTicketLinkValue', tsTicketLink)
      urlParameters = urlParameters.replace('tsTicketPriorityValue', tsTicketPriority)
      urlParameters = urlParameters.replace('tsTicketUrgencyValue', tsTicketUrgency)

      DataOutputStream wr = new DataOutputStream(conn.getOutputStream());
      wr.writeBytes(urlParameters);
      wr.flush();
      wr.close();

      int responseCode = conn.getResponseCode();
      if (responseCode == HttpURLConnection.HTTP_OK) 
			{
				InputStream response = conn.getInputStream();
				String content = new java.util.Scanner(response).useDelimiter("\\A").next();
				if(content!=null) 
				{
          appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter -- GET Response ${content.toString()} ###\n")
          appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
				}
			}
      else if ((responseCode.toInteger() != 201) && (responseCode.toInteger() != 202)) 
      {
        InputStream response = conn.getInputStream();
        String content = new java.util.Scanner(response).useDelimiter("\\A").next();
        appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter -- Post request failed ${content} ###\n")
        appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
      }
      conn.disconnect();
    } 
    catch (Exception e) 
    {
      def scError = e.toString()
      appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter -- Post request failed ${scError} ###\n")
      appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
    }
  }

  def updateOpsgenieState(def event, def apiKey)
  {
    /* https://community.microfocus.com/it_ops_mgt/ops_bdg/f/itrc-162/348722/event-getcustomattribute-and-event-getnode-do-not-work-in-the-logfileadapter-sample-script */
    def timestamp = new Date()
    String eventId = "${event.id.toString()}"
    String eventState = "${event.state.toString()}"
    String opsgenieAlertId = "${event.getType().toString()}"

    appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter updateOpsgenieState() called ###\n")
    appendLogfile("\nOpsgenie alertId: ${opsgenieAlertId}\n")
    appendLogfile("\nState of eventId: ${eventId} -> ${eventState}\n")
    appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")

    try 
    {
      String genericURL = "https://api.eu.opsgenie.com/v2/alerts/alert_id/close?identifierType=id";
      genericURL = genericURL.replace('alert_id', opsgenieAlertId)

      URL url = new URL(genericURL);
      Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress("1.2.3.4", 8080));
      HttpURLConnection conn = (HttpURLConnection) url.openConnection(proxy);

      conn.setRequestProperty("Authorization", "GenieKey " + "${apiKey.toString()}");
      conn.setRequestMethod("POST");
      conn.setRequestProperty("Content-Type", "application/json; utf-8");
      conn.setDoOutput(true);

      String urlParameters = """{
        "user": "Smile Webhook",
        "source": "OBM",
        "note": "Closing the alarm since EventId: ${eventId} in OBM got resolved/closed"
      }"""

      DataOutputStream wr = new DataOutputStream(conn.getOutputStream());
      wr.writeBytes(urlParameters);
      wr.flush();
      wr.close();

      int responseCode = conn.getResponseCode();
      if (responseCode == HttpURLConnection.HTTP_OK) 
			{
				InputStream response = conn.getInputStream();
				String content = new java.util.Scanner(response).useDelimiter("\\A").next();
				if(content!=null) 
				{
          appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter -- GET Response ${content.toString()} ###\n")
          appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
				}
			}
      else if ((responseCode.toInteger() != 201) && (responseCode.toInteger() != 202)) 
      {
        InputStream response = conn.getInputStream();
        String content = new java.util.Scanner(response).useDelimiter("\\A").next();
        appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter -- Post request failed ${content} ###\n")
        appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
      }
      conn.disconnect();
    } 
    catch (Exception e) 
    {
      def scError = e.toString()
      appendLogfile("\n### [INFO]: ${timestamp.toString()}: OpsgenieBacksyncAdapter -- Post request failed ${scError} ###\n")
      appendLogfile("\n-------------------------------------------------------------------------------------------------------\n")
    }

  }

  def appendLogfile(Object message)
  {
    if (!m_logfileDir.exists())
      m_logfileDir.mkdirs()

    if (!m_logfile.exists())
      m_logfile.createNewFile()

    m_logfile.append(message)
  }
}