#Author: Vidyasagar Machani
#Date: 2023-10-04
#Purpose: This script scans MIB files in the same directory as the script itself, extracts OBJECT-TYPE definitions, and prints them.
#Version: 3.0

#!/usr/bin/python

import re
import os
import glob # Module for finding files matching a pattern

# This list will store ALL object names found, including duplicates initially
all_object_names = [] # Renamed from object_names for clarity in this step

# Determine the directory where the script is located
script_dir = os.path.dirname(os.path.abspath(__file__))

# Compile the regex pattern once for efficiency
pattern = re.compile(r"^\s*(\w+)\s+OBJECT-TYPE")

# Construct the search pattern for .mib files within the script's directory
search_pattern = os.path.join(script_dir, '*.mib')

print(f"Scanning script's directory: '{script_dir}' for '.mib' files...")

# Use glob to find all files matching the pattern
mib_files = glob.glob(search_pattern)

if not mib_files:
    print("No '.mib' files found in the script's directory.")
else:
    print(f"Found {len(mib_files)} MIB file(s). Processing...")
    print("-" * 30)

    # Iterate through each found MIB file
    for mib_file_path in mib_files:
        mib_filename = os.path.basename(mib_file_path)
        print(f"--- Processing file: {mib_filename} ---")
        # Keep track of names found just in this file to print the '(No ...)' message
        found_in_this_file = False

        try:
            # Use 'with open' for proper file handling
            with open(mib_file_path, 'r', encoding='utf-8', errors='ignore') as file1:
                for line_num, line in enumerate(file1, 1):
                    stripped_line = line.strip()
                    match = pattern.search(stripped_line)

                    if match:
                        object_name = match.group(1)
                        print(object_name) # Print immediately
                        all_object_names.append(object_name) # Add to the global list
                        found_in_this_file = True

            if not found_in_this_file:
                 print("(No OBJECT-TYPE definitions found in this file)")
            print("-" * 30)

        except Exception as e:
            print(f"An error occurred while processing '{mib_filename}': {e}")
            print("-" * 30)


print("Finished processing all MIB files.")
print("=" * 30) # Add a separator for clarity

# --- Deduplication and Final Print ---
if all_object_names: # Check if the list is not empty
    # 1. Convert the list to a set to automatically remove duplicates
    unique_names_set = set(all_object_names)
    # 2. Convert the set back to a list
    unique_names_list = list(unique_names_set)
    # 3. Optional: Sort the unique list alphabetically
    unique_names_list.sort()

    print(f"\nUnique OBJECT-TYPE names found ({len(unique_names_list)}):")
    # Print each unique name, one per line
    #for name in unique_names_list:
    #    print(name)
    # Or, if you prefer the list format:
    print(unique_names_list)

else:
    # Handle the case where no object names were found at all
    print("\nNo OBJECT-TYPE definitions were found in any MIB file.")